-- =============================================
-- RAVENA TURISMO - BANCO DE DADOS COMPLETO
-- Sistema: office.ravenaturismo.com.br
-- Versão: 1.0 Final
-- =============================================

-- Criar e selecionar banco
CREATE DATABASE IF NOT EXISTS ravena_turismo 
CHARACTER SET utf8mb4 
COLLATE utf8mb4_unicode_ci;

USE ravena_turismo;

-- Remover tabelas se existirem
DROP TABLE IF EXISTS historico_orcamento;
DROP TABLE IF EXISTS solicitacoes_orcamento;
DROP TABLE IF EXISTS pacote_galeria;
DROP TABLE IF EXISTS pacote_valores;
DROP TABLE IF EXISTS pacote_checklist;
DROP TABLE IF EXISTS pacote_roteiro;
DROP TABLE IF EXISTS pacote_inclusos;
DROP TABLE IF EXISTS pacotes;
DROP TABLE IF EXISTS categorias;
DROP TABLE IF EXISTS banners;
DROP TABLE IF EXISTS configuracoes_sistema;
DROP TABLE IF EXISTS usuarios;

-- =============================================
-- TABELA: USUARIOS
-- =============================================
CREATE TABLE usuarios (
    id INT PRIMARY KEY AUTO_INCREMENT,
    nome VARCHAR(100) NOT NULL,
    email VARCHAR(100) UNIQUE NOT NULL,
    senha VARCHAR(255) NOT NULL,
    nivel ENUM('admin', 'editor') DEFAULT 'editor',
    ativo TINYINT(1) DEFAULT 1,
    ultimo_acesso TIMESTAMP NULL,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_email (email),
    INDEX idx_ativo (ativo)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- TABELA: CATEGORIAS
-- =============================================
CREATE TABLE categorias (
    id INT PRIMARY KEY AUTO_INCREMENT,
    nome VARCHAR(100) NOT NULL,
    slug VARCHAR(100) UNIQUE NOT NULL,
    icone VARCHAR(50) DEFAULT '📦',
    descricao TEXT,
    ordem INT DEFAULT 0,
    ativo TINYINT(1) DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_ativo (ativo),
    INDEX idx_ordem (ordem)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- TABELA: PACOTES
-- =============================================
CREATE TABLE pacotes (
    id INT PRIMARY KEY AUTO_INCREMENT,
    categoria_id INT NOT NULL,
    titulo VARCHAR(200) NOT NULL,
    slug VARCHAR(200) UNIQUE NOT NULL,
    subtitulo VARCHAR(300),
    descricao_curta TEXT,
    descricao_completa LONGTEXT,
    foto_capa VARCHAR(255),
    duracao VARCHAR(50),
    destino VARCHAR(100),
    ativo TINYINT(1) DEFAULT 1,
    destaque TINYINT(1) DEFAULT 0,
    visualizacoes INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (categoria_id) REFERENCES categorias(id) ON DELETE CASCADE,
    INDEX idx_categoria (categoria_id),
    INDEX idx_slug (slug),
    INDEX idx_ativo (ativo),
    INDEX idx_destaque (destaque),
    INDEX idx_ativo_destaque (ativo, destaque)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- TABELA: PACOTE_INCLUSOS
-- =============================================
CREATE TABLE pacote_inclusos (
    id INT PRIMARY KEY AUTO_INCREMENT,
    pacote_id INT NOT NULL,
    tipo ENUM('incluso', 'nao_incluso') NOT NULL,
    item TEXT NOT NULL,
    ordem INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (pacote_id) REFERENCES pacotes(id) ON DELETE CASCADE,
    INDEX idx_pacote_tipo (pacote_id, tipo),
    INDEX idx_ordem (ordem)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- TABELA: PACOTE_ROTEIRO
-- =============================================
CREATE TABLE pacote_roteiro (
    id INT PRIMARY KEY AUTO_INCREMENT,
    pacote_id INT NOT NULL,
    dia INT NOT NULL,
    titulo VARCHAR(200),
    descricao TEXT NOT NULL,
    ordem INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (pacote_id) REFERENCES pacotes(id) ON DELETE CASCADE,
    INDEX idx_pacote (pacote_id),
    INDEX idx_dia (dia)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- TABELA: PACOTE_CHECKLIST
-- =============================================
CREATE TABLE pacote_checklist (
    id INT PRIMARY KEY AUTO_INCREMENT,
    pacote_id INT NOT NULL,
    item VARCHAR(255) NOT NULL,
    categoria VARCHAR(50) DEFAULT 'Geral',
    ordem INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (pacote_id) REFERENCES pacotes(id) ON DELETE CASCADE,
    INDEX idx_pacote (pacote_id),
    INDEX idx_categoria (categoria)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- TABELA: PACOTE_VALORES
-- =============================================
CREATE TABLE pacote_valores (
    id INT PRIMARY KEY AUTO_INCREMENT,
    pacote_id INT NOT NULL,
    moeda ENUM('BRL', 'USD', 'EUR') NOT NULL DEFAULT 'BRL',
    valor_unitario DECIMAL(10,2) NOT NULL,
    tipo_acomodacao VARCHAR(100) DEFAULT 'Standard',
    observacao TEXT,
    ativo TINYINT(1) DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (pacote_id) REFERENCES pacotes(id) ON DELETE CASCADE,
    INDEX idx_pacote (pacote_id),
    INDEX idx_moeda (moeda)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- TABELA: PACOTE_GALERIA
-- =============================================
CREATE TABLE pacote_galeria (
    id INT PRIMARY KEY AUTO_INCREMENT,
    pacote_id INT NOT NULL,
    tipo ENUM('imagem', 'video') NOT NULL DEFAULT 'imagem',
    arquivo VARCHAR(255) NOT NULL,
    legenda VARCHAR(255),
    youtube_id VARCHAR(20),
    ordem INT DEFAULT 0,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (pacote_id) REFERENCES pacotes(id) ON DELETE CASCADE,
    INDEX idx_pacote_tipo (pacote_id, tipo),
    INDEX idx_ordem (ordem)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- TABELA: BANNERS
-- =============================================
CREATE TABLE banners (
    id INT PRIMARY KEY AUTO_INCREMENT,
    titulo VARCHAR(200) NOT NULL,
    subtitulo VARCHAR(300),
    imagem VARCHAR(255) NOT NULL,
    link VARCHAR(255),
    link_texto VARCHAR(50) DEFAULT 'Ver mais',
    ordem INT DEFAULT 0,
    ativo TINYINT(1) DEFAULT 1,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    INDEX idx_ativo (ativo),
    INDEX idx_ordem (ordem)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- TABELA: SOLICITACOES_ORCAMENTO
-- =============================================
CREATE TABLE solicitacoes_orcamento (
    id INT PRIMARY KEY AUTO_INCREMENT,
    nome_completo VARCHAR(150) NOT NULL,
    email VARCHAR(150) NOT NULL,
    telefone VARCHAR(20) NOT NULL,
    whatsapp VARCHAR(20),
    pacote_id INT NULL,
    pacote_nome VARCHAR(200),
    destino_interesse VARCHAR(200),
    data_ida DATE NULL,
    data_volta DATE NULL,
    num_adultos INT DEFAULT 1,
    num_criancas INT DEFAULT 0,
    num_bebes INT DEFAULT 0,
    tipo_acomodacao VARCHAR(100),
    orcamento_estimado VARCHAR(50),
    observacoes TEXT,
    como_conheceu VARCHAR(100),
    origem_formulario VARCHAR(50) DEFAULT 'site',
    url_origem VARCHAR(255),
    status ENUM('novo', 'em_analise', 'proposta_enviada', 'fechado', 'perdido', 'cancelado') DEFAULT 'novo',
    prioridade ENUM('baixa', 'media', 'alta', 'urgente') DEFAULT 'media',
    responsavel_id INT NULL,
    enviado_n8n TINYINT(1) DEFAULT 0,
    data_envio_n8n TIMESTAMP NULL,
    webhook_response TEXT,
    ip_cliente VARCHAR(45),
    user_agent TEXT,
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP,
    FOREIGN KEY (pacote_id) REFERENCES pacotes(id) ON DELETE SET NULL,
    FOREIGN KEY (responsavel_id) REFERENCES usuarios(id) ON DELETE SET NULL,
    INDEX idx_status (status),
    INDEX idx_email (email),
    INDEX idx_telefone (telefone),
    INDEX idx_created (created_at),
    INDEX idx_enviado_n8n (enviado_n8n),
    INDEX idx_pacote (pacote_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- TABELA: HISTORICO_ORCAMENTO
-- =============================================
CREATE TABLE historico_orcamento (
    id INT PRIMARY KEY AUTO_INCREMENT,
    orcamento_id INT NOT NULL,
    usuario_id INT NULL,
    acao VARCHAR(100) NOT NULL,
    descricao TEXT,
    status_anterior VARCHAR(50),
    status_novo VARCHAR(50),
    created_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP,
    FOREIGN KEY (orcamento_id) REFERENCES solicitacoes_orcamento(id) ON DELETE CASCADE,
    FOREIGN KEY (usuario_id) REFERENCES usuarios(id) ON DELETE SET NULL,
    INDEX idx_orcamento (orcamento_id)
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- TABELA: CONFIGURACOES_SISTEMA
-- =============================================
CREATE TABLE configuracoes_sistema (
    id INT PRIMARY KEY AUTO_INCREMENT,
    chave VARCHAR(100) UNIQUE NOT NULL,
    valor TEXT,
    descricao VARCHAR(255),
    tipo ENUM('texto', 'numero', 'boolean', 'json') DEFAULT 'texto',
    updated_at TIMESTAMP DEFAULT CURRENT_TIMESTAMP ON UPDATE CURRENT_TIMESTAMP
) ENGINE=InnoDB DEFAULT CHARSET=utf8mb4 COLLATE=utf8mb4_unicode_ci;

-- =============================================
-- DADOS INICIAIS
-- =============================================

-- Usuários (senha: Ravena@2025)
INSERT INTO usuarios (nome, email, senha, nivel, ativo) VALUES
('Administrador Ravena', 'admin@ravenaturismo.com.br', '$2y$10$7aWJEWs5Y8WGHgvPH.4uje3PXXqjPxQp7hvKy3k.zb9L7YxT0qE4G', 'admin', 1),
('Editor Ravena', 'editor@ravenaturismo.com.br', '$2y$10$7aWJEWs5Y8WGHgvPH.4uje3PXXqjPxQp7hvKy3k.zb9L7YxT0qE4G', 'editor', 1);

-- Categorias
INSERT INTO categorias (nome, slug, icone, descricao, ordem) VALUES
('Aéreo', 'aereo', '✈️', 'Pacotes com passagens aéreas incluídas', 1),
('Terrestre', 'terrestre', '🚌', 'Viagens terrestres de ônibus ou van', 2),
('Experiências', 'experiencias', '🎭', 'Experiências únicas e inesquecíveis', 3),
('Roteiros', 'roteiros', '🗺️', 'Roteiros personalizados completos', 4);

-- Banners
INSERT INTO banners (titulo, subtitulo, imagem, link, ordem, ativo) VALUES
('Descubra o Mundo com a Ravena Turismo', 'Experiências inesquecíveis te aguardam', 'banner-exemplo.jpg', 'categoria.php?slug=aereo', 1, 1),
('Pacotes Especiais para 2025', 'Descontos incríveis para as melhores destinações', 'banner-exemplo-2.jpg', NULL, 2, 1);

-- Pacote Exemplo: Paris
INSERT INTO pacotes (categoria_id, titulo, slug, subtitulo, descricao_curta, descricao_completa, foto_capa, duracao, destino, ativo, destaque) VALUES
(1, 'Paris - Cidade Luz', 'paris-cidade-luz', 'Uma viagem inesquecível pela capital francesa', 
'Conheça Paris, a cidade mais romântica do mundo! Tour pela Torre Eiffel, Museu do Louvre, Champs-Élysées e muito mais.', 
'Descubra Paris em uma viagem completa de 7 dias, com hospedagem em hotéis 4 estrelas, passeios guiados e experiências gastronômicas autênticas. Inclui visitas aos principais pontos turísticos, tempo livre para compras e noites mágicas na Cidade Luz.',
'paris-capa.jpg', '7 dias / 6 noites', 'Paris, França', 1, 1);

SET @pacote_id = LAST_INSERT_ID();

-- Inclusos
INSERT INTO pacote_inclusos (pacote_id, tipo, item, ordem) VALUES
(@pacote_id, 'incluso', 'Passagem aérea Brasil - Paris (ida e volta)', 1),
(@pacote_id, 'incluso', '6 noites de hospedagem em hotel 4 estrelas', 2),
(@pacote_id, 'incluso', 'Café da manhã todos os dias', 3),
(@pacote_id, 'incluso', 'Transfer aeroporto - hotel - aeroporto', 4),
(@pacote_id, 'incluso', 'Tour guiado pela Torre Eiffel', 5),
(@pacote_id, 'incluso', 'Ingresso para o Museu do Louvre', 6),
(@pacote_id, 'incluso', 'Passeio de barco pelo Rio Sena', 7),
(@pacote_id, 'incluso', 'Guia brasileiro durante toda a viagem', 8),
(@pacote_id, 'incluso', 'Seguro viagem internacional', 9);

-- Não Inclusos
INSERT INTO pacote_inclusos (pacote_id, tipo, item, ordem) VALUES
(@pacote_id, 'nao_incluso', 'Visto consular (se necessário)', 1),
(@pacote_id, 'nao_incluso', 'Refeições não mencionadas no roteiro', 2),
(@pacote_id, 'nao_incluso', 'Bebidas alcoólicas', 3),
(@pacote_id, 'nao_incluso', 'Despesas pessoais (telefonemas, lavanderia)', 4),
(@pacote_id, 'nao_incluso', 'Gorjetas para guias e motoristas', 5),
(@pacote_id, 'nao_incluso', 'Passeios opcionais não mencionados', 6);

-- Roteiro
INSERT INTO pacote_roteiro (pacote_id, dia, titulo, descricao, ordem) VALUES
(@pacote_id, 1, 'Chegada em Paris', 'Chegada no Aeroporto Charles de Gaulle. Recepção pela equipe Ravena e transfer para o hotel. Tempo livre para descanso. À noite, jantar de boas-vindas.', 1),
(@pacote_id, 2, 'Paris Clássica', 'Café da manhã. Visita guiada à Torre Eiffel. Almoço livre. À tarde, Champs-Élysées e Arco do Triunfo. Noite livre.', 2),
(@pacote_id, 3, 'Arte e Cultura', 'Visita ao Museu do Louvre com guia. Tarde livre para compras ou passeio opcional a Versalhes.', 3),
(@pacote_id, 4, 'Rio Sena e Montmartre', 'Passeio de barco pelo Rio Sena. Visita a Montmartre e Sacré-Cœur. Noite livre.', 4),
(@pacote_id, 5, 'Dia Livre', 'Dia livre para explorar Paris. Sugestões: Disney, outlet ou bairros históricos.', 5),
(@pacote_id, 6, 'Últimas Descobertas', 'Manhã livre. À tarde, jantar especial com vista para Torre Eiffel iluminada.', 6),
(@pacote_id, 7, 'Retorno ao Brasil', 'Café da manhã. Check-out e transfer para aeroporto. Fim dos serviços.', 7);

-- Checklist
INSERT INTO pacote_checklist (pacote_id, item, categoria, ordem) VALUES
(@pacote_id, 'Passaporte válido (mínimo 6 meses)', 'Documentos', 1),
(@pacote_id, 'Visto Schengen (se aplicável)', 'Documentos', 2),
(@pacote_id, 'Carteira de vacinação atualizada', 'Documentos', 3),
(@pacote_id, 'Cópia de todos os documentos', 'Documentos', 4),
(@pacote_id, 'Roupas de frio (inverno)', 'Bagagem', 5),
(@pacote_id, 'Roupas leves (verão)', 'Bagagem', 6),
(@pacote_id, 'Calçados confortáveis', 'Bagagem', 7),
(@pacote_id, 'Adaptador de tomada europeu', 'Eletrônicos', 8),
(@pacote_id, 'Carregadores', 'Eletrônicos', 9),
(@pacote_id, 'Euros em espécie', 'Financeiro', 10),
(@pacote_id, 'Cartão de crédito internacional', 'Financeiro', 11),
(@pacote_id, 'Medicamentos pessoais', 'Saúde', 12),
(@pacote_id, 'Protetor solar', 'Saúde', 13);

-- Valores
INSERT INTO pacote_valores (pacote_id, moeda, valor_unitario, tipo_acomodacao, observacao) VALUES
(@pacote_id, 'BRL', 12500.00, 'Apartamento Duplo', 'Valor por pessoa em apartamento duplo'),
(@pacote_id, 'BRL', 15800.00, 'Apartamento Single', 'Valor por pessoa em apartamento individual'),
(@pacote_id, 'BRL', 11200.00, 'Apartamento Triplo', 'Valor por pessoa em apartamento triplo'),
(@pacote_id, 'USD', 2450.00, 'Apartamento Duplo', 'Valor por pessoa em apartamento duplo'),
(@pacote_id, 'EUR', 2280.00, 'Apartamento Duplo', 'Valor por pessoa em apartamento duplo');

-- Galeria (placeholders - você vai adicionar imagens reais)
INSERT INTO pacote_galeria (pacote_id, tipo, arquivo, legenda, ordem) VALUES
(@pacote_id, 'imagem', 'paris-torre-eiffel.jpg', 'Torre Eiffel iluminada', 1),
(@pacote_id, 'imagem', 'paris-louvre.jpg', 'Museu do Louvre', 2),
(@pacote_id, 'imagem', 'paris-champs.jpg', 'Champs-Élysées', 3),
(@pacote_id, 'imagem', 'paris-sena.jpg', 'Rio Sena', 4),
(@pacote_id, 'imagem', 'paris-montmartre.jpg', 'Montmartre', 5),
(@pacote_id, 'imagem', 'paris-arco.jpg', 'Arco do Triunfo', 6);

-- Configurações do Sistema
INSERT INTO configuracoes_sistema (chave, valor, descricao, tipo) VALUES
('n8n_webhook_url', '', 'URL do webhook n8n para orçamentos', 'texto'),
('n8n_webhook_ativo', '0', 'Ativar envio automático para n8n', 'boolean'),
('n8n_webhook_secret', '', 'Chave secreta para autenticação', 'texto'),
('email_notificacao_orcamento', 'contato@ravenaturismo.com.br', 'Email para notificações', 'texto');

-- =============================================
-- CONFIRMAÇÃO
-- =============================================
SELECT '✅ Banco de dados criado com sucesso!' AS status;
SELECT CONCAT('📊 Total de tabelas: ', COUNT(*)) AS info FROM information_schema.tables WHERE table_schema = 'ravena_turismo';
SELECT CONCAT('👥 Usuários criados: ', COUNT(*)) AS info FROM usuarios;
SELECT CONCAT('📦 Categorias: ', COUNT(*)) AS info FROM categorias;
SELECT CONCAT('✈️ Pacotes: ', COUNT(*)) AS info FROM pacotes;