<?php
/**
 * Funções Auxiliares
 */

function sanitize($data) {
    return htmlspecialchars(strip_tags(trim($data)), ENT_QUOTES, 'UTF-8');
}

function generateSlug($text) {
    $text = preg_replace('~[^\pL\d]+~u', '-', $text);
    $text = iconv('utf-8', 'us-ascii//TRANSLIT', $text);
    $text = preg_replace('~[^-\w]+~', '', $text);
    $text = trim($text, '-');
    $text = preg_replace('~-+~', '-', $text);
    $text = strtolower($text);
    return empty($text) ? 'n-a' : $text;
}

function formatCurrency($value, $currency = 'BRL') {
    $symbols = [
        'BRL' => 'R$ ',
        'USD' => '$ ',
        'EUR' => '€ '
    ];
    return $symbols[$currency] . number_format($value, 2, ',', '.');
}

function getCurrencyRates() {
    $cacheFile = sys_get_temp_dir() . '/currency_rates.json';
    $cacheTime = 3600;
    
    if (file_exists($cacheFile) && (time() - filemtime($cacheFile)) < $cacheTime) {
        return json_decode(file_get_contents($cacheFile), true);
    }
    
    try {
        $url = 'https://economia.awesomeapi.com.br/json/last/USD-BRL,EUR-BRL';
        $response = file_get_contents($url);
        $data = json_decode($response, true);
        
        $rates = [
            'USD' => floatval($data['USDBRL']['bid']),
            'EUR' => floatval($data['EURBRL']['bid']),
            'updated' => date('d/m/Y H:i')
        ];
        
        file_put_contents($cacheFile, json_encode($rates));
        return $rates;
    } catch (Exception $e) {
        return [
            'USD' => 5.00,
            'EUR' => 5.50,
            'updated' => date('d/m/Y H:i')
        ];
    }
}

function redirect($url) {
    header("Location: $url");
    exit;
}

function setFlash($type, $message) {
    $_SESSION['flash'] = ['type' => $type, 'message' => $message];
}

function getFlash() {
    if (isset($_SESSION['flash'])) {
        $flash = $_SESSION['flash'];
        unset($_SESSION['flash']);
        return $flash;
    }
    return null;
}

/**
 * Envia orçamento para n8n
 */
function enviarParaN8n($orcamento_id, $dados) {
    $db = getDB();
    
    try {
        $stmt = $db->prepare("SELECT valor FROM configuracoes_sistema WHERE chave = 'n8n_webhook_url'");
        $stmt->execute();
        $config = $stmt->fetch();
        $webhookUrl = $config['valor'] ?? '';
        
        if (empty($webhookUrl)) {
            return false;
        }
        
        $stmt = $db->prepare("SELECT valor FROM configuracoes_sistema WHERE chave = 'n8n_webhook_ativo'");
        $stmt->execute();
        $ativo = $stmt->fetch();
        
        if ($ativo['valor'] !== '1') {
            return false;
        }
        
        $payload = [
            'orcamento_id' => $orcamento_id,
            'timestamp' => date('Y-m-d H:i:s'),
            'cliente' => [
                'nome' => $dados['nome_completo'],
                'email' => $dados['email'],
                'telefone' => $dados['telefone'],
                'whatsapp' => $dados['whatsapp']
            ],
            'viagem' => [
                'destino' => $dados['destino_interesse'],
                'pacote_id' => $dados['pacote_id'],
                'pacote_nome' => $dados['pacote_nome'],
                'data_ida' => $dados['data_ida'],
                'data_volta' => $dados['data_volta'],
                'passageiros' => [
                    'adultos' => $dados['num_adultos'],
                    'criancas' => $dados['num_criancas'],
                    'bebes' => $dados['num_bebes']
                ],
                'acomodacao' => $dados['tipo_acomodacao'],
                'orcamento' => $dados['orcamento_estimado']
            ],
            'observacoes' => $dados['observacoes'],
            'origem' => [
                'formulario' => $dados['origem_formulario'],
                'como_conheceu' => $dados['como_conheceu'],
                'url' => $dados['url_origem']
            ],
            'links' => [
                'admin' => ADMIN_URL . '/orcamentos/detalhes.php?id=' . $orcamento_id,
                'responder' => 'mailto:' . $dados['email'],
                'whatsapp' => 'https://wa.me/' . preg_replace('/\D/', '', $dados['telefone'])
            ]
        ];
        
        $ch = curl_init($webhookUrl);
        curl_setopt($ch, CURLOPT_RETURNTRANSFER, true);
        curl_setopt($ch, CURLOPT_POST, true);
        curl_setopt($ch, CURLOPT_POSTFIELDS, json_encode($payload));
        curl_setopt($ch, CURLOPT_HTTPHEADER, [
            'Content-Type: application/json',
            'User-Agent: RavenaTurismo/1.0'
        ]);
        curl_setopt($ch, CURLOPT_TIMEOUT, 10);
        
        $response = curl_exec($ch);
        $httpCode = curl_getinfo($ch, CURLINFO_HTTP_CODE);
        curl_close($ch);
        
        $stmt = $db->prepare("
            UPDATE solicitacoes_orcamento 
            SET enviado_n8n = 1, 
                data_envio_n8n = NOW(),
                webhook_response = ?
            WHERE id = ?
        ");
        $stmt->execute([
            json_encode(['http_code' => $httpCode, 'response' => $response]),
            $orcamento_id
        ]);
        
        return $httpCode >= 200 && $httpCode < 300;
        
    } catch (Exception $e) {
        error_log("Erro n8n: " . $e->getMessage());
        return false;
    }
}