<?php
require_once '../includes/config.php';
require_once '../includes/database.php';
require_once '../includes/functions.php';

$db = getDB();

// Pegar pacote_id se vier da URL
$pacote_id = isset($_GET['pacote']) ? (int)$_GET['pacote'] : null;
$pacote = null;

if ($pacote_id) {
    $stmt = $db->prepare("SELECT * FROM pacotes WHERE id = ? AND ativo = 1");
    $stmt->execute([$pacote_id]);
    $pacote = $stmt->fetch();
}

// Buscar todas as categorias para o select
$stmtCategorias = $db->query("SELECT * FROM categorias WHERE ativo = 1 ORDER BY ordem");
$categorias = $stmtCategorias->fetchAll();

// Processar envio do formulário
$mensagem = null;
$tipo_mensagem = null;

if ($_SERVER['REQUEST_METHOD'] === 'POST') {
    try {
        // Validar e sanitizar dados
        $dados = [
            'nome_completo' => sanitize($_POST['nome_completo']),
            'email' => filter_var($_POST['email'], FILTER_SANITIZE_EMAIL),
            'telefone' => sanitize($_POST['telefone']),
            'whatsapp' => sanitize($_POST['whatsapp'] ?? ''),
            'pacote_id' => !empty($_POST['pacote_id']) ? (int)$_POST['pacote_id'] : null,
            'pacote_nome' => sanitize($_POST['pacote_nome'] ?? ''),
            'destino_interesse' => sanitize($_POST['destino_interesse']),
            'data_ida' => !empty($_POST['data_ida']) ? $_POST['data_ida'] : null,
            'data_volta' => !empty($_POST['data_volta']) ? $_POST['data_volta'] : null,
            'num_adultos' => (int)$_POST['num_adultos'],
            'num_criancas' => (int)$_POST['num_criancas'],
            'num_bebes' => (int)$_POST['num_bebes'],
            'tipo_acomodacao' => sanitize($_POST['tipo_acomodacao']),
            'orcamento_estimado' => sanitize($_POST['orcamento_estimado']),
            'observacoes' => sanitize($_POST['observacoes']),
            'como_conheceu' => sanitize($_POST['como_conheceu']),
            'origem_formulario' => 'site',
            'url_origem' => $_SERVER['HTTP_REFERER'] ?? '',
            'ip_cliente' => $_SERVER['REMOTE_ADDR'],
            'user_agent' => $_SERVER['HTTP_USER_AGENT']
        ];
        
        // Validações básicas
        if (empty($dados['nome_completo']) || empty($dados['email']) || empty($dados['telefone'])) {
            throw new Exception('Preencha todos os campos obrigatórios.');
        }
        
        if (!filter_var($dados['email'], FILTER_VALIDATE_EMAIL)) {
            throw new Exception('Email inválido.');
        }
        
        // Inserir no banco
        $stmt = $db->prepare("
            INSERT INTO solicitacoes_orcamento (
                nome_completo, email, telefone, whatsapp,
                pacote_id, pacote_nome, destino_interesse,
                data_ida, data_volta,
                num_adultos, num_criancas, num_bebes,
                tipo_acomodacao, orcamento_estimado, observacoes,
                como_conheceu, origem_formulario, url_origem,
                ip_cliente, user_agent
            ) VALUES (
                :nome_completo, :email, :telefone, :whatsapp,
                :pacote_id, :pacote_nome, :destino_interesse,
                :data_ida, :data_volta,
                :num_adultos, :num_criancas, :num_bebes,
                :tipo_acomodacao, :orcamento_estimado, :observacoes,
                :como_conheceu, :origem_formulario, :url_origem,
                :ip_cliente, :user_agent
            )
        ");
        
        $stmt->execute($dados);
        $orcamento_id = $db->lastInsertId();
        
        // ========== ENVIAR PARA n8n ==========
        enviarParaN8n($orcamento_id, $dados);
        
        $mensagem = "Orçamento solicitado com sucesso! Entraremos em contato em breve.";
        $tipo_mensagem = "success";
        
        // Limpar formulário
        $_POST = [];
        
    } catch (Exception $e) {
        $mensagem = "Erro ao enviar solicitação: " . $e->getMessage();
        $tipo_mensagem = "error";
    }
}
?>
<!DOCTYPE html>
<html lang="pt-BR">
<head>
    <meta charset="UTF-8">
    <meta name="viewport" content="width=device-width, initial-scale=1.0">
    <title>Solicitar Orçamento - Ravena Turismo</title>
    <link href="https://fonts.googleapis.com/css2?family=Inter:wght@400;500;600;700&display=swap" rel="stylesheet">
    <link rel="stylesheet" href="assets/css/style.css">
    <style>
        .form-container {
            max-width: 800px;
            margin: 3rem auto;
            padding: 2rem;
            background: white;
            border-radius: 16px;
            box-shadow: 0 4px 20px rgba(0,0,0,0.1);
        }
        
        .form-header {
            text-align: center;
            margin-bottom: 2rem;
        }
        
        .form-header h1 {
            font-size: 2rem;
            color: var(--color-primary);
            margin-bottom: 0.5rem;
        }
        
        .form-header p {
            color: var(--color-text-light);
        }
        
        .form-group {
            margin-bottom: 1.5rem;
        }
        
        .form-group label {
            display: block;
            font-weight: 600;
            margin-bottom: 0.5rem;
            color: var(--color-text);
        }
        
        .form-group label .required {
            color: #ef4444;
        }
        
        .form-group input,
        .form-group select,
        .form-group textarea {
            width: 100%;
            padding: 0.875rem;
            border: 1px solid var(--color-border);
            border-radius: 8px;
            font-size: 1rem;
            font-family: inherit;
            transition: border-color 0.2s;
        }
        
        .form-group input:focus,
        .form-group select:focus,
        .form-group textarea:focus {
            outline: none;
            border-color: var(--color-primary);
            box-shadow: 0 0 0 3px rgba(30, 64, 175, 0.1);
        }
        
        .form-row {
            display: grid;
            grid-template-columns: repeat(auto-fit, minmax(200px, 1fr));
            gap: 1rem;
        }
        
        .alert {
            padding: 1rem;
            border-radius: 8px;
            margin-bottom: 1.5rem;
            font-weight: 500;
        }
        
        .alert-success {
            background: #d1fae5;
            color: #065f46;
            border: 1px solid #6ee7b7;
        }
        
        .alert-error {
            background: #fee2e2;
            color: #991b1b;
            border: 1px solid #fca5a5;
        }
        
        .btn-submit {
            width: 100%;
            padding: 1rem;
            background: var(--color-primary);
            color: white;
            border: none;
            border-radius: 8px;
            font-size: 1.125rem;
            font-weight: 600;
            cursor: pointer;
            transition: all 0.3s;
        }
        
        .btn-submit:hover {
            background: var(--color-primary-dark);
            transform: translateY(-2px);
            box-shadow: 0 4px 12px rgba(30, 64, 175, 0.3);
        }
        
        .package-info {
            background: #eff6ff;
            padding: 1rem;
            border-radius: 8px;
            margin-bottom: 2rem;
            border-left: 4px solid var(--color-primary);
        }
        
        .package-info h3 {
            color: var(--color-primary);
            margin-bottom: 0.5rem;
        }
    </style>
</head>
<body>
    <div class="form-container">
        <div class="form-header">
            <h1>✈️ Solicitar Orçamento</h1>
            <p>Preencha o formulário e entraremos em contato em breve!</p>
        </div>
        
        <?php if ($mensagem): ?>
        <div class="alert alert-<?= $tipo_mensagem ?>">
            <?= $mensagem ?>
        </div>
        <?php endif; ?>
        
        <?php if ($pacote): ?>
        <div class="package-info">
            <h3>📦 Pacote Selecionado</h3>
            <p><strong><?= htmlspecialchars($pacote['titulo']) ?></strong></p>
            <p><?= htmlspecialchars($pacote['destino']) ?> • <?= htmlspecialchars($pacote['duracao']) ?></p>
        </div>
        <?php endif; ?>
        
        <form method="POST" action="">
            <?php if ($pacote): ?>
            <input type="hidden" name="pacote_id" value="<?= $pacote['id'] ?>">
            <input type="hidden" name="pacote_nome" value="<?= htmlspecialchars($pacote['titulo']) ?>">
            <?php endif; ?>
            
            <!-- Dados Pessoais -->
            <h2 style="margin-bottom: 1rem;">👤 Seus Dados</h2>
            
            <div class="form-group">
                <label>Nome Completo <span class="required">*</span></label>
                <input type="text" name="nome_completo" required 
                       value="<?= $_POST['nome_completo'] ?? '' ?>"
                       placeholder="Digite seu nome completo">
            </div>
            
            <div class="form-row">
                <div class="form-group">
                    <label>Email <span class="required">*</span></label>
                    <input type="email" name="email" required 
                           value="<?= $_POST['email'] ?? '' ?>"
                           placeholder="seu@email.com">
                </div>
                
                <div class="form-group">
                    <label>Telefone <span class="required">*</span></label>
                    <input type="tel" name="telefone" required 
                           value="<?= $_POST['telefone'] ?? '' ?>"
                           placeholder="(00) 00000-0000">
                </div>
            </div>
            
            <div class="form-group">
                <label>WhatsApp (se diferente do telefone)</label>
                <input type="tel" name="whatsapp" 
                       value="<?= $_POST['whatsapp'] ?? '' ?>"
                       placeholder="(00) 00000-0000">
            </div>
            
            <!-- Dados da Viagem -->
            <h2 style="margin: 2rem 0 1rem;">✈️ Dados da Viagem</h2>
            
            <div class="form-group">
                <label>Destino de Interesse <span class="required">*</span></label>
                <input type="text" name="destino_interesse" required 
                       value="<?= $_POST['destino_interesse'] ?? ($pacote['destino'] ?? '') ?>"
                       placeholder="Ex: Paris, Dubai, Caribe...">
            </div>
            
            <div class="form-row">
                <div class="form-group">
                    <label>Data de Ida (aproximada)</label>
                    <input type="date" name="data_ida" 
                           value="<?= $_POST['data_ida'] ?? '' ?>">
                </div>
                
                <div class="form-group">
                    <label>Data de Volta (aproximada)</label>
                    <input type="date" name="data_volta" 
                           value="<?= $_POST['data_volta'] ?? '' ?>">
                </div>
            </div>
            
            <div class="form-row">
                <div class="form-group">
                    <label>Adultos <span class="required">*</span></label>
                    <input type="number" name="num_adultos" min="1" value="<?= $_POST['num_adultos'] ?? 1 ?>" required>
                </div>
                
                <div class="form-group">
                    <label>Crianças (2-12 anos)</label>
                    <input type="number" name="num_criancas" min="0" value="<?= $_POST['num_criancas'] ?? 0 ?>">
                </div>
                
                <div class="form-group">
                    <label>Bebês (0-2 anos)</label>
                    <input type="number" name="num_bebes" min="0" value="<?= $_POST['num_bebes'] ?? 0 ?>">
                </div>
            </div>
            
            <div class="form-row">
                <div class="form-group">
                    <label>Tipo de Acomodação</label>
                    <select name="tipo_acomodacao">
                        <option value="">Selecione...</option>
                        <option value="Standard">Standard</option>
                        <option value="Duplo">Apartamento Duplo</option>
                        <option value="Single">Apartamento Single</option>
                        <option value="Triplo">Apartamento Triplo</option>
                        <option value="Luxo">Luxo / Premium</option>
                    </select>
                </div>
                
                <div class="form-group">
                    <label>Orçamento Estimado</label>
                    <select name="orcamento_estimado">
                        <option value="">Selecione...</option>
                        <option value="Até R$ 5.000">Até R$ 5.000</option>
                        <option value="R$ 5.000 - R$ 10.000">R$ 5.000 - R$ 10.000</option>
                        <option value="R$ 10.000 - R$ 20.000">R$ 10.000 - R$ 20.000</option>
                        <option value="R$ 20.000 - R$ 30.000">R$ 20.000 - R$ 30.000</option>
                        <option value="Acima de R$ 30.000">Acima de R$ 30.000</option>
                        <option value="Flexível">Flexível</option>
                    </select>
                </div>
            </div>
            
            <div class="form-group">
                <label>Observações / Preferências</label>
                <textarea name="observacoes" rows="4" 
                          placeholder="Conte-nos mais sobre sua viagem ideal..."><?= $_POST['observacoes'] ?? '' ?></textarea>
            </div>
            
            <div class="form-group">
                <label>Como conheceu a Ravena Turismo?</label>
                <select name="como_conheceu">
                    <option value="">Selecione...</option>
                    <option value="Google">Google</option>
                    <option value="Instagram">Instagram</option>
                    <option value="Facebook">Facebook</option>
                    <option value="WhatsApp">WhatsApp</option>
                    <option value="Indicação">Indicação de amigo/familiar</option>
                    <option value="Outro">Outro</option>
                </select>
            </div>
            
            <button type="submit" class="btn-submit">
                📧 Enviar Solicitação
            </button>
        </form>
    </div>
    
    <script>
        // Máscaras de telefone
        document.querySelectorAll('input[type="tel"]').forEach(input => {
            input.addEventListener('input', function(e) {
                let value = e.target.value.replace(/\D/g, '');
                if (value.length <= 11) {
                    value = value.replace(/^(\d{2})(\d{5})(\d{4})/, '($1) $2-$3');
                    value = value.replace(/^(\d{2})(\d{4})(\d{0,4})/, '($1) $2-$3');
                }
                e.target.value = value;
            });
        });
    </script>
</body>
</html>